<?php
/*
Plugin Name: Workable Careers Embed
Plugin URI: https://www.workable.com/
Description: A plugin that embeds Workable job listings of your career page into a Wordpress page.
Version: 1.0
Author: Workable HR
Author URI: https://www.workable.com/
*/

define('WORKABLE_JOBS_PLUGIN_SLUG', 'workable-careers-embed');
define('WORKABLE_JOBS_PLUGIN_SETTINGS_URL', admin_url('options-general.php?page=workable-jobs-settings'));

# Settings page of the plugin
function workable_jobs_settings_page()
{
    if (!current_user_can('manage_options')) {
        return;
    }
?>
    <div class="wrap">
        <h1>Workable Careers Settings</h1>
        <form method="post" action="options.php">
            <?php settings_fields('workable_jobs_options'); ?>
            <?php do_settings_sections('workable_jobs_options'); ?>
            <table class="form-table">
                <tbody>
                    <tr>
                        <th scope="row"><label for="workable_id">Workable ID:</label></th>
                        <td>
                            <input type="text" name="workable_id" id="workable_id" value="<?php echo get_option('workable_id', ''); ?>" placeholder="Enter your Workable ID here" required />
                            <p class="description">Enter your Workable account ID here to embed your job listings from Workable.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="listing">Initial Listing:</label></th>
                        <td>
                            <select name="workable_jobs_listing" id="listing">
                                <option value="jobs" <?php selected(get_option('workable_jobs_listing'), 'jobs'); ?>>Jobs</option>
                                <option value="locations" <?php selected(get_option('workable_jobs_listing'), 'locations'); ?>>List by location</option>
                                <option value="departments" <?php selected(get_option('workable_jobs_listing'), 'departments'); ?>>List by department</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="detail">View Detail:</label></th>
                        <td>
                            <select name="workable_jobs_detail" id="detail">
                                <option value="titles" <?php selected(get_option('workable_jobs_detail'), 'titles'); ?>>Show only titles</option>
                                <option value="descriptions" <?php selected(get_option('workable_jobs_detail'), 'descriptions'); ?>>Show full descriptions</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="zoom">Location Detail:</label></th>
                        <td>
                            <select name="workable_jobs_zoom" id="zoom">
                                <option value="country" <?php selected(get_option('workable_jobs_zoom'), 'country'); ?>>Country</option>
                                <option value="state" <?php selected(get_option('workable_jobs_zoom'), 'state'); ?>>State/Region</option>
                                <option value="city" <?php selected(get_option('workable_jobs_zoom'), 'city'); ?>>City</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="grouping">Grouping:</label></th>
                        <td>
                            <select name="workable_jobs_grouping" id="grouping">
                                <option value="none" <?php selected(get_option('workable_jobs_grouping'), 'none'); ?>>No grouping</option>
                                <option value="location" <?php selected(get_option('workable_jobs_grouping'), 'location'); ?>>Group by location</option>
                                <option value="department" <?php selected(get_option('workable_jobs_grouping'), 'department'); ?>>Group by department</option>
                            </select>
                        </td>
                    </tr>
                </tbody>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
<?php
}
function sanitize_int_field($field_value) {
    $sanitized_value = intval($field_value);
    return $sanitized_value;
}

function workable_jobs_register_settings()
{
    register_setting('workable_jobs_options', 'workable_id', 'sanitize_int_field');
    register_setting('workable_jobs_options', 'workable_jobs_listing', 'sanitize_text_field');
    register_setting('workable_jobs_options', 'workable_jobs_detail', 'sanitize_text_field');
    register_setting('workable_jobs_options', 'workable_jobs_zoom', 'sanitize_text_field');
    register_setting('workable_jobs_options', 'workable_jobs_grouping', 'sanitize_text_field');
}

add_action('admin_init', 'workable_jobs_register_settings');


# Setup a warning message if the Workable ID is not set
function workable_jobs_plugin_init()
{
    if (!get_option('workable_id', false)) {
        add_action('admin_notices', 'workable_jobs_admin_notice');
    }
}

function workable_jobs_admin_notice()
{
    echo '<div class="notice notice-error"><p><strong>Error: Workable ID is not set. Please set the Workable ID option in the plugin <a href="' . WORKABLE_JOBS_PLUGIN_SETTINGS_URL . '">settings</a>.</strong></p></div>';
}

add_action('plugins_loaded', 'workable_jobs_plugin_init');

# Set up link on the admin menu for plugin's setting page
function workable_jobs_add_admin_settings_link()
{
    add_submenu_page(
        'options-general.php',
        'Workable Careers Settings',
        'Workable Careers Settings',
        'manage_options',
        'workable-jobs-settings',
        'workable_jobs_settings_page'
    );
}
add_action('admin_menu', 'workable_jobs_add_admin_settings_link');


function workable_jobs_add_plugin_settings_link($links)
{
    $settings_link = '<a href="' . WORKABLE_JOBS_PLUGIN_SETTINGS_URL . '">Settings</a>';
    array_push($links, $settings_link);
    return $links;
}

add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'workable_jobs_add_plugin_settings_link');


# Shortcode to display the job listings
function workable_careers_shortcode($atts)
{
    $atts = shortcode_atts(array(
        'id' => ''
    ), $atts);

    $workable_id = !defined($atts['id']) || empty($atts['id']) ? get_option('workable_id') : $atts['id'];

    $listing_option = get_option('workable_jobs_listing', 'jobs');
    $listing = ($listing_option == 'jobs') ? 'jobs' : (($listing_option == 'locations') ? 'locations' : 'departments');

    $detail_option = get_option('workable_jobs_detail', 'titles');
    $detail = ($detail_option == 'titles') ? 'titles' : 'descriptions';

    $zoom_option = get_option('workable_jobs_zoom', 'country');
    $zoom = ($zoom_option == 'country') ? 'country' : (($zoom_option == 'state') ? 'state' : 'city');

    $grouping_option = get_option('workable_jobs_grouping', 'none');
    $grouping = ($grouping_option == 'none') ? 'none' : (($grouping_option == 'location') ? 'locations' : 'departments');

    ob_start();
?>
    <script src='https://www.workable.com/assets/embed.js' type='text/javascript'></script>
    <script type='text/javascript' charset='utf-8'>
        whr(document).ready(function() {
            whr_embed(<?php echo $workable_id; ?>, {
                detail: '<?php echo $detail; ?>',
                base: '<?php echo $listing; ?>',
                zoom: '<?php echo $zoom; ?>',
                grouping: '<?php echo $grouping; ?>'
            });
        });
    </script>
    <div class='check' id='whr_embed_hook'></div>
<?php
    return ob_get_clean();
}
add_shortcode('workable_careers', 'workable_careers_shortcode');


# Uninstalling plugin - clean up settings
function workable_jobs_uninstall()
{
    delete_option('workable_id');
    delete_option('workable_jobs_listing');
    delete_option('workable_jobs_detail');
    delete_option('workable_jobs_zoom');
    delete_option('workable_jobs_grouping');
}

register_uninstall_hook(__FILE__, 'workable_jobs_uninstall');
